/* ================================================
   OCAMPO KLEVER — Landing Page Script
   Canva Pro Activation Service
   ================================================ */

// ── WhatsApp Configuration ──
const WHATSAPP_NUMBER = '593960705007'; // +593 96 070 5007

// ── Mensajes contextuales por botón ──
const WA_MESSAGES = {
  // Navbar
  'nav-cta': '👋 Hola Klever, vi tu página y me interesa activar Canva Pro. ¿Cómo funciona?',

  // Hero principal
  'hero-cta-primary': '🚀 Hola Klever, quiero activar Canva Pro ahora. ¿Qué necesito hacer? Mi correo de Canva es: ________.',

  // Sección Beneficios
  'beneficios-cta': '✨ Hola Klever, vi los beneficios de Canva Pro y quiero activarlo. ¿Cuáles son los pasos?',

  // Precios — Plan $5
  'pricing-cta-5': '💳 Hola Klever, quiero el plan de Canva Pro por $5 (1 año). Mi correo de Canva es: ________. ¿Cómo activo?',

  // Precios — Plan $8
  'pricing-cta-8': '💎 Hola Klever, quiero el plan de Canva Pro por $8 (2 años). Mi correo de Canva es: ________. ¿Cómo activo?',

  // FAQ
  'faq-cta': '❓ Hola Klever, tengo algunas dudas sobre Canva Pro antes de activarlo. ¿Me puedes ayudar?',

  // CTA Final
  'final-cta-btn': '🔥 Hola Klever, quiero activar Canva Pro YA. Dime qué necesitas y empezamos.',

  // Botón flotante
  'wa-float-btn': '💬 Hola Klever, estoy en tu página y tengo una consulta rápida sobre Canva Pro.'
};

// Mensaje por defecto
const WA_DEFAULT = encodeURIComponent(
  'Hola Klever, quiero Canva Pro. Mi correo de Canva es: ________. Quiero el plan de $5 (1 año) / $8 (2 años).'
);

function getWhatsAppURL(el) {
  const id = el.id || '';
  const plan = el.getAttribute('data-wa') || '';
  let msg;

  if (WA_MESSAGES[id]) {
    msg = encodeURIComponent(WA_MESSAGES[id]);
  } else if (plan === '5') {
    msg = encodeURIComponent('💳 Hola Klever, quiero el plan de Canva Pro por $5 (1 año). Mi correo de Canva es: ________.');
  } else if (plan === '8') {
    msg = encodeURIComponent('💎 Hola Klever, quiero el plan de Canva Pro por $8 (2 años). Mi correo de Canva es: ________.');
  } else {
    msg = WA_DEFAULT;
  }

  return `https://wa.me/${WHATSAPP_NUMBER}?text=${msg}`;
}

// Set all WhatsApp links
function initWhatsAppLinks() {
  document.querySelectorAll('[data-wa]').forEach(el => {
    el.href = getWhatsAppURL(el);
    el.target = '_blank';
    el.rel = 'noopener noreferrer';
  });
}

// ── Particles ──
function createParticles() {
  const container = document.getElementById('particles-container');
  if (!container) return;
  const count = 50;

  for (let i = 0; i < count; i++) {
    const particle = document.createElement('div');
    particle.classList.add('particle');

    // Random position
    particle.style.left = Math.random() * 100 + '%';

    // Random size
    const size = Math.random() * 3 + 1;
    particle.style.width = size + 'px';
    particle.style.height = size + 'px';

    // Random animation duration and delay
    const duration = Math.random() * 15 + 10;
    const delay = Math.random() * 15;
    particle.style.animationDuration = duration + 's';
    particle.style.animationDelay = delay + 's';

    // Random opacity
    particle.style.setProperty('--max-opacity', (Math.random() * 0.5 + 0.1).toString());

    container.appendChild(particle);
  }
}

// ── Navbar Scroll Effect ──
function initNavbar() {
  const navbar = document.querySelector('.navbar');
  if (!navbar) return;

  const onScroll = () => {
    if (window.scrollY > 50) {
      navbar.classList.add('scrolled');
    } else {
      navbar.classList.remove('scrolled');
    }
  };

  window.addEventListener('scroll', onScroll, { passive: true });
  onScroll();
}

// ── Mobile Menu ──
function initMobileMenu() {
  const hamburger = document.querySelector('.hamburger');
  const navLinks = document.querySelector('.navbar-links');
  if (!hamburger || !navLinks) return;

  hamburger.addEventListener('click', () => {
    hamburger.classList.toggle('active');
    navLinks.classList.toggle('active');
  });

  // Close menu on link click
  navLinks.querySelectorAll('a').forEach(link => {
    link.addEventListener('click', () => {
      hamburger.classList.remove('active');
      navLinks.classList.remove('active');
    });
  });
}

// ── Intersection Observer for Reveal ──
function initRevealAnimations() {
  const elements = document.querySelectorAll('.reveal, .reveal-left, .reveal-right');

  const observer = new IntersectionObserver((entries) => {
    entries.forEach(entry => {
      if (entry.isIntersecting) {
        entry.target.classList.add('visible');
      }
    });
  }, {
    threshold: 0.1,
    rootMargin: '0px 0px -50px 0px'
  });

  elements.forEach(el => observer.observe(el));
}

// ── Animated Counters ──
function initCounters() {
  const counters = document.querySelectorAll('[data-count]');

  const animateCounter = (el) => {
    const target = parseInt(el.getAttribute('data-count'), 10);
    const suffix = el.getAttribute('data-suffix') || '';
    const duration = 2000;
    const startTime = performance.now();

    const update = (currentTime) => {
      const elapsed = currentTime - startTime;
      const progress = Math.min(elapsed / duration, 1);

      // Ease-out quad
      const eased = 1 - (1 - progress) * (1 - progress);
      const current = Math.floor(eased * target);

      el.textContent = current.toLocaleString() + suffix;

      if (progress < 1) {
        requestAnimationFrame(update);
      }
    };

    requestAnimationFrame(update);
  };

  const observer = new IntersectionObserver((entries) => {
    entries.forEach(entry => {
      if (entry.isIntersecting && !entry.target.dataset.animated) {
        entry.target.dataset.animated = 'true';
        animateCounter(entry.target);
      }
    });
  }, { threshold: 0.5 });

  counters.forEach(el => observer.observe(el));
}

// ── Flip Cards (Mobile Tap) ──
function initFlipCards() {
  const cards = document.querySelectorAll('.flip-card');
  const isTouchDevice = 'ontouchstart' in window || navigator.maxTouchPoints > 0;

  if (isTouchDevice) {
    cards.forEach(card => {
      card.addEventListener('click', (e) => {
        // Close others
        cards.forEach(c => {
          if (c !== card) c.classList.remove('flipped');
        });
        card.classList.toggle('flipped');
      });
    });
  }
}

// ── FAQ Accordion ──
function initFAQ() {
  const items = document.querySelectorAll('.faq-item');

  items.forEach(item => {
    const question = item.querySelector('.faq-question');

    question.addEventListener('click', () => {
      const isActive = item.classList.contains('active');

      // Close all
      items.forEach(i => i.classList.remove('active'));

      // Toggle current
      if (!isActive) {
        item.classList.add('active');
      }
    });
  });
}

// ── Smooth Scroll ──
function initSmoothScroll() {
  document.querySelectorAll('a[href^="#"]').forEach(anchor => {
    anchor.addEventListener('click', function (e) {
      const href = this.getAttribute('href');
      if (href === '#') return;

      e.preventDefault();
      const target = document.querySelector(href);
      if (target) {
        target.scrollIntoView({
          behavior: 'smooth',
          block: 'start'
        });
      }
    });
  });
}

// ── Stagger animation for grid items ──
function initStaggerAnimations() {
  const grids = document.querySelectorAll('.benefits-grid, .pricing-grid, .steps-grid');

  grids.forEach(grid => {
    const children = grid.children;
    Array.from(children).forEach((child, index) => {
      child.style.transitionDelay = `${index * 100}ms`;
    });
  });
}

// ── Initialize Everything ──
document.addEventListener('DOMContentLoaded', () => {
  initWhatsAppLinks();
  createParticles();
  initNavbar();
  initMobileMenu();
  initRevealAnimations();
  initCounters();
  initFlipCards();
  initFAQ();
  initSmoothScroll();
  initStaggerAnimations();
});
